<?php

namespace invCustomFields\Inc\Claz;

use DateInterval;
use DateTime;
use Exception;
use NumberFormatter;

use Inc\Claz\DbField;
use Inc\Claz\DomainId;
use Inc\Claz\FromStmt;
use Inc\Claz\FunctionStmt;
use Inc\Claz\Invoice as OrigInvoice;
use Inc\Claz\Join;
use Inc\Claz\OrderBy;
use Inc\Claz\Payment;
use Inc\Claz\PaymentType;
use Inc\Claz\PaymentWarehouse;
use Inc\Claz\Select;
use Inc\Claz\Util;
use Inc\Claz\WhereClause;
use Inc\Claz\WhereItem;

use Inc\Claz\PdoDbException;

    /**
     * Class Invoice
     * @package InvCustomFields\Inc\Claz
     */
class Invoice extends OrigInvoice
{
    /**
     * Retrieve all the invoices with using a having value if specified.
     * @param array|string $having Can be:
     *          <b>string:</b> One of the values defined in the buildHavings() method.
     *          <b>array:</b> Associative array with an array element:
     *              Ex: array("date_between" => array($start_date, $end_date)), or
     *                  array("date_between" => array($start_date, $end_date),
     *                        "real" => array());
     *          <b>array:</b> Sequential array containing one or more associative arrays:
     *              Ex: array(array("date_between" => array($start_date, $end_date),
     *                        array("real" => array());
     * @param string $sort field to order by (optional).
     * @param string $dir order by direction, "asc" or "desc" - (optional).
     * @param bool $manageTable true if select for manage.tpl table; false (default) if not.
     * @param bool $includeWarehouse true if should include warehouse info with all selected, false if not.\
     * @param int $invoiceDisplayDays If specified, the days in the past invoices will be retrieved for.
     * @return array Invoices retrieved.
     * @throws PdoDbException
     */
    public static function getAllWithHavings(array|string $having, string $sort = "", string $dir = "", bool $manageTable = false,
                                             bool $includeWarehouse = false, int $invoiceDisplayDays = 0): array
    {
        global $pdoDb;

        if (!$manageTable) {
            return Invoice::getAllWithHavings($having, $sort, $dir, $manageTable, $includeWarehouse, $invoiceDisplayDays);
        }

        if (!empty($having)) {
            try {
                if (is_array($having)) {
                    foreach ($having as $key => $value) {
                        if (is_int($key) && is_array($value)) {
                            foreach ($value as $key2 => $value2) {
                                $pdoDb->setHavings(Invoice::buildHavings($key2, $value2));
                            }
                        } else {
                            $pdoDb->setHavings(Invoice::buildHavings($key, $value));
                        }
                    }
                } elseif (is_string($having)) {
                    $pdoDb->setHavings(Invoice::buildHavings($having));
                } else {
                    // Will be caught below
                    throw new PdoDbException("Invalid having parameter passed - " . print_r($having, true));
                }
            } catch (PdoDbException $pde) {
                error_log("Invoice::getAllWithHavings() - Error: " . $pde->getMessage());
                throw $pde;
            }
        }

        return self::manageTableInfo($invoiceDisplayDays);
    }

    /**
     * Minimize the amount of data returned to the manage table.
     * @param int $invoiceDisplayDays Number of days in the past to limit invoices retrieved; 0 means no limit.
     * @return array Data for the manage table rows.
     * @throws PdoDbException
     */
    public static function manageTableInfo(int $invoiceDisplayDays = 0): array
    {
        global $config, $LANG;

        $readOnly = isset($_SESSION['role_name']) && $_SESSION['role_name'] == 'customer';
        $rows = Invoice::getInvoices(null, '', '', false, $invoiceDisplayDays);
        $tableRows = [];
        foreach ($rows as $row) {
            // @formatter:off
            if ($readOnly) {
                $invEdit = '';
                $invPymt = '';
            } else {
                $invEdit = "<a class='index_table' title='{$LANG['edit']} {$row['index_id']}' " .
                    "href='index.php?module=invoices&amp;view=edit&amp;id={$row['id']}'>" .
                    "<img src='images/edit.png' class='action' alt='edit'>" .
                    "</a>";

                $invPymt = "<!-- Alternatively: The Owing column can have the link on the amount instead of the payment icon code here -->";
                if ($row['status'] == ENABLED && $row['owing'] > 0) {
                    $invPymt .= "<!-- Real Invoice Has Owing - Process payment -->" .
                        "<a title='{$LANG['processPayment']} {$row['index_id']}' class='index_table' " .
                        "href='index.php?module=payments&amp;view=process&amp;id={$row['id']}&amp;op=pay_selected_invoice'>" .
                        "<img src='images/money_dollar.png' class='action' alt='payment'/>" .
                        "</a>";
                } elseif ($row['status'] == ENABLED) {
                    $invPymt .= "<!-- Real Invoice Payment Details if not Owing (get different color payment icon) -->" .
                        "<a title='{$LANG['processPayment']} {$row['index_id']}' class='index_table' " .
                        "href='index.php?module=payments&amp;view=view&amp;ac_inv_id={$row['id']}'>" .
                        "<img src='images/money_dollar.png' class='action' alt='payment' />" .
                        "</a>";
                } else {
                    $invPymt .= "<!-- Draft Invoice Just Image to occupy space till blank or greyed out icon becomes available -->" .
                        "<img src='images/money_dollar.png' class='action' alt='payment' />";
                }
            }

            $action = "<a class='index_table' title='{$LANG['quickViewTooltip']} {$row['index_id']}' " .
                "href='index.php?module=invoices&amp;view=quickView&amp;id={$row['id']}'>" .
                "<img src='images/view.png' class='action' alt='view' />" .
                "</a>" .
                $invEdit .
                "<a class='index_table desktopOnly' title='{$LANG['printPreviewTooltip']} {$row['index_id']}' target='_blank' " .
                "href='index.php?module=export&amp;view=invoice&amp;id={$row['id']}&amp;format=print'>" .
                "<img src='images/printer.png' class='action' alt='print' />" .
                "</a>" .
                "<a class='invoice_export_dialog desktopOnly' id='btnShowSimple' title='{$LANG['exportUc']} {$row['index_id']}' " .
                "href='#' data-row-num='{$row['id']}' data-spreadsheet='{$config['exportSpreadsheet']}' " .
                "data-wordprocessor='{$config['exportWordProcessor']}'>" .
                "<img src='images/page_white_acrobat.png' class='action' alt='spreadsheet'/>" .
                "</a>" .
                $invPymt .
                "<a title='{$LANG['email']} {$row['index_id']}' class='index_table desktopOnly' " .
                "href='index.php?module=invoices&amp;view=email&amp;stage=1&amp;id={$row['id']}'>" .
                "<img src='images/mail-message-new.png' class='action' alt='email' />" .
                "</a>";

            $pattern = '/^(.*)_(.*)$/';
            $tableRows[] = [
                'action' => $action,
                'index_id' => $row['index_id'],
                'biller' => $row['biller'],
                'customer' => $row['customer'],
                'preference' => $row['preference'],
                'date' => $row['date'],
                'total' => $row['total'],
                'owing' => isset($row['status']) ? $row['owing'] : '',
                'aging' => $row['aging'] ?? '',
                'currency_code' => $row['currency_code'],
                'locale' => preg_replace($pattern,'$1-$2', $row['locale']),
                'custom_field1' => $row['custom_field1'],
                'custom_field2' => $row['custom_field2'],
                'custom_field3' => $row['custom_field3'],
                'custom_field4' => $row['custom_field4']
            ];
        }
        return $tableRows;
    }
}